// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.pay.echo.config;

import com.google.samples.pgp.KeyManagementException;
import com.google.samples.pgp.KeyManager;
import com.google.samples.pgp.PgpKeyManager;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.logging.Logger;

import javax.annotation.PostConstruct;

import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;

@Configuration
public class KeyConfig {

  private static final Logger LOGGER = Logger.getLogger(KeyConfig.class.getName());

  /**
   * Adds the private and public keys, as well as the passphrase to the key manager on application
   * startup.
   */
  @PostConstruct
  private static void addKeys() {
    ResourceLoader resourceLoader = new DefaultResourceLoader();
    addPublicKeys(resourceLoader);
    addPrivateKeyAndPassphrase(resourceLoader);
  }

  /**
   * Adds the public keys to the Key Manager.
   *
   * @param resourceLoader Loads the existing keys from the resources directory.
   */
  private static void addPublicKeys(ResourceLoader resourceLoader) {
    KeyManager keyManager = PgpKeyManager.getInstance();
      try {
        keyManager.addPublicKeys(resourceLoader.getResource("classpath:publickey/publicKey1.gpg")
            .getInputStream());
        // To add a second public key, uncomment the lines below and add your second
        // key as 'publicKey2.gpg' in the resources folder.
        // keyManager.addPublicKeys(resourceLoader.getResource("classpath:publickey/publicKey2.gpg")
        //     .getInputStream());
        LOGGER.info("Successfully added public key(s)!");
      } catch (IOException | KeyManagementException e) {
        LOGGER.severe("Error adding public key credentials");
        throw new RuntimeException("Error adding public key");
      }
  }

  /**
   * Adds the private keys and passphrase to the Key Manager.
   *
   * @param resourceLoader Loads the existing keys from the resources directory.
   */
  private static void addPrivateKeyAndPassphrase(ResourceLoader resourceLoader) {
    KeyManager keyManager = PgpKeyManager.getInstance();
      try {
        Resource passphraseResource1 =
            resourceLoader.getResource("classpath:passphrase/passphrase1.txt");
        BufferedReader passPhraseBufferedReader1 =
            new BufferedReader(new InputStreamReader(passphraseResource1.getInputStream()));
        keyManager.addSecretKeys(
            resourceLoader.getResource("classpath:privatekey/privateKey1.gpg").getInputStream(),
            passPhraseBufferedReader1.readLine().toCharArray());

        // To add a second private key, uncomment the lines below and add your private key as
        // privateKey2.gpg and add your passphrase as passphrase2.txt in the resources folder.
        // Resource passphraseResource2 =
        //     resourceLoader.getResource("classpath:passphrase/passphrase2.txt");
        // BufferedReader passPhraseBufferedReader2 =
        //     new BufferedReader(new InputStreamReader(passphraseResource2.getInputStream()));
        // keyManager.addSecretKeys(
        //     resourceLoader.getResource("classpath:privatekey/privateKey2.gpg").getInputStream(),
        //     passPhraseBufferedReader2.readLine().toCharArray());
        LOGGER.info("Successfully added private key credentials!");
      } catch (IOException | KeyManagementException e) {
        LOGGER.severe("Error adding private key credentials");
        throw new RuntimeException("Error adding private key credentials");
      }
  }
}